///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Mike Heck (Dec 2008)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

/*----------------------------------------------------------------------

Example program.
Shows a simple volume shader for multi-channel data.

This test data was created by segmenting the tiny data set SYN_64.vol
into 3 "channels".  Every combination of the three channels is visible
in the resulting rendering.

This example only shows volume rendering and assumes 3 channels of
data.  To do slice rendering and/or handle a different number of
channels requires a slightly different shader program and corresponding
different setup of the shader parameters.

This example completely replaces the default VolumeViz fragment shader
main program (to keep things as simple as possible).  As a result it
does not support VolumeViz advanced rendering options such as jittering,
edge color and boundary opacity.  There is a separate example for this.
----------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <LDM/nodes/SoMultiDataSeparator.h> 
#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

static SoTransferFunction* makeColorMap(SbColor color, float gamma, int id);

static SoVolumeRenderingQuality* makeVolumeShader();

////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Initialize and create the window
  Widget myWindow = SoXt::init(argv[0]);
  SoVolumeRendering::init();
  InventorMedical::init();

  SoRef<SoSeparator> root = new SoSeparator();

  // Camera
  SoPerspectiveCamera* camera = new SoPerspectiveCamera();
    root->addChild( camera );

  // Assemble the volume rendering scene graph.
  // Note that we use a MultiDataSeparator node to hold multiple data
  // sets that will be combined in a single rendering node.
  SoMultiDataSeparator* volSep = new SoMultiDataSeparator();
    root->addChild( volSep );

  // ---------- Load volume data ----------
  // Each channel is in a separate file.
  // Note that each volume must have a unique dataSetId.
  // We also increase the tile size so each channel can be loaded as a block.
  SoVolumeData* volData0 = new SoVolumeData();
    volData0->fileName = "$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalMultiChannel/Channel1.am";
    volData0->ldmResourceParameters.getValue()->tileDimension.setValue(256,256,256);
    volData0->dataSetId = 1;
    volSep->addChild( volData0 );

  SoVolumeData* volData1 = new SoVolumeData();
    volData1->fileName = "$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalMultiChannel/Channel2.am";
    volData1->ldmResourceParameters.getValue()->tileDimension.setValue(256,256,256);
    volData1->dataSetId = 2;
    volSep->addChild( volData1 );

  SoVolumeData* volData2 = new SoVolumeData();
    volData2->fileName = "$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalMultiChannel/Channel3.am";
    volData2->ldmResourceParameters.getValue()->tileDimension.setValue(256,256,256);
    volData2->dataSetId = 3;
    volSep->addChild( volData2 );
  
  // ---------- Map actual range of data values ----------
  // This test data only uses values 0..192, so we will not use all the
  // values in the colormap unless we explicitly set the data range.
  // Note:
  // 1. Each volume data node must have its own data range node.
  // 2. The dataRangeId must be the same as the dataSetId of the
  //    corresponding volume data node.
  //
  // Note that the getMinMax query is "free" for LDM format data,
  // because the min and max values are stored in the file header.
  // Custom readers may also support this query.  But many of the
  // standard volume readers have to load the entire data set from
  // disk to answer this query and this can be very time consuming.
  // If using such a reader, when possible, preprocess the data and
  // store the min/max values in some sort of "info" file that the
  // application can load quickly and avoid calling getMinMax().
  double dataMin, dataMax;
  SoDataRange* volRange0 = new SoDataRange();
    volData0->getMinMax(dataMin, dataMax);
    volRange0->min = dataMin;
    volRange0->max = dataMax;
    volRange0->dataRangeId = 1;

  SoDataRange* volRange1 = new SoDataRange();
    volData1->getMinMax(dataMin, dataMax);
    volRange1->min = dataMin;
    volRange1->max = dataMax;
    volRange1->dataRangeId = 2;

  SoDataRange* volRange2 = new SoDataRange();
    volData2->getMinMax(dataMin, dataMax);
    volRange2->min = dataMin;
    volRange2->max = dataMax;
    volRange2->dataRangeId = 3;

  // ---------- Make color map for each channel ----------
  // Typically for multichannel rendering we will create a colormap
  // with a linear color ramp, i.e. black -> full color, and a gamma
  // curve alpha ramp.
  // Note that each volume must have a unique transferFunctionId,
  // which generally should be the same as the corresponding volume
  // data node's dataSetId.
  // BLUE
  SoTransferFunction* volTF0 = makeColorMap(SbColor(0, 0, 1), 2.39f, 1);
  volSep->addChild( volTF0 );
  // GREEN
  SoTransferFunction* volTF1 = makeColorMap(SbColor(0, 1, 0), 2.39f, 2);
  volSep->addChild( volTF1 );
  // RED
  SoTransferFunction* volTF2 = makeColorMap(SbColor(1, 0, 0), 2.39f, 3);
  volSep->addChild( volTF2 );

  // Volume quality node.
  // Note this node is both a shader node and a property node.
  SoVolumeRenderingQuality* volShader = makeVolumeShader();
    volShader->interpolateOnMove = TRUE;
    volShader->preIntegrated = TRUE;
    volSep->addChild(volShader);

  // Render node
  SoVolumeRender* volRender = new SoVolumeRender;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volRender->numSlicesControl  = SoVolumeRender::AUTOMATIC;
    volSep->addChild( volRender );

  // Display bounding box of volume
  root->addChild(MedicalHelper::createBoundingBox(volData1->extent.getValue()));

  // OIV Logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Note
  TextBox* text = new TextBox();
    text->position.setValue(0, -0.99f, 0); // Normalized device coords -1..1
    text->alignmentH = TextBox::CENTER;
    text->alignmentV = TextBox::BOTTOM;
    text->addLine("Three channels (red/green/blue) combined on GPU.");
    root->addChild(text);

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTransparencyType(SoGLRenderAction::NO_SORT);
    myViewer->setTitle("Multichannel rendering");
    myViewer->setSize( MedicalHelper::exampleWindowSize() );
    myViewer->setDecoration(FALSE);

  // Adjust camera (flipping orientation of volume in this case).
  myViewer->getCamera()->orientation.setValue( SbVec3f(1, 0, 0), (float)M_PI );
  myViewer->viewAll();
  myViewer->saveHomePosition();

  // Run then cleanup
  myViewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


////////////////////////////////////////////////////////////////////////
//
// Make the volume shader node
//
// Builds the FragmentShader and VolumeShader objects and sets up
// the basic parameters.
//
// Note: This version assumes there are 3 channels of data and the
//       data will be rendered using SoVolumeRender.

SoVolumeRenderingQuality* makeVolumeShader()
{
  // Load fragment shader program
  SoFragmentShader* fragmentShader = new SoFragmentShader();
  fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/Medical/Rendering/Multiple Volumes/medicalMultiChannel/MultiChannel_color_frag.glsl");

  // Create uniform parameters allowing shader to access textures
  //
  // Transfer function is in texture unit 0 (default)
  SoShaderParameter1i *paramTex0 = new SoShaderParameter1i;
  paramTex0->name = "transfer";
  paramTex0->value.setValue(0);

  // Volume data is in texture units 1, 2 and 3
  SoShaderParameter1i *paramTex1 = new SoShaderParameter1i;
  paramTex1->name = "voldata1";
  paramTex1->value.setValue(1);
  SoShaderParameter1i *paramTex2 = new SoShaderParameter1i;
  paramTex2->name = "voldata2";
  paramTex2->value.setValue(2);
  SoShaderParameter1i *paramTex3 = new SoShaderParameter1i;
  paramTex3->name = "voldata3";
  paramTex3->value.setValue(3);

  fragmentShader->parameter.set1Value(0, paramTex0);
  fragmentShader->parameter.set1Value(1, paramTex1);
  fragmentShader->parameter.set1Value(2, paramTex2);
  fragmentShader->parameter.set1Value(3, paramTex3);

  // Initialize the volume shader
  // Note in this version we completely replace the default VolumeViz
  // fragment shader main program (to keep the example simple).
  SoVolumeRenderingQuality* volShader = new SoVolumeRenderingQuality;
  volShader->shaderObject.set1Value(SoVolumeRenderingQuality::FRAGMENT_COMPUTE_COLOR, fragmentShader);

  // Specify shader should be applied to SoVolumeRender nodes.
  // (Name is a little deceptive because you might think that FALSE means
  // for BOTH volumes and slices, but it's really one or the other.)
  volShader->forVolumeOnly = TRUE;

  return volShader;
}

////////////////////////////////////////////////////////////////////////
//
// Make a multi-channel colormap (transfer function)
//
// This function creates a colormap appropriate for (some) multi-channel
// volume data.  It contains a linear color ramp, i.e. the minimum data
// value is mapped to black and the maximum data value is mapped to the
// full color.  The alpha ramp is a gamma curve, so the higher the gamma
// value the more strongly low valued voxels are suppressed.
//
// The colormaps must have unique id numbers, starting at zero.

SoTransferFunction *
makeColorMap(SbColor color, float gamma, int id)
{
  SoTransferFunction *pTF = new SoTransferFunction();
  pTF->transferFunctionId = id;

  // Color map will contain 256 RGBA values -> 1024 float values.
  // setNum pre-allocates memory so we can edit the field directly.
  pTF->colorMap.setNum(256 * 4);

  float R = color[0];
  float G = color[1];
  float B = color[2];

  // Get an edit pointer, assign values, then finish editing.
  float *p = pTF->colorMap.startEditing();
  for (int i = 0; i < 256; ++i) {
    float factor = (float)i / 255;
    *p++ = R * factor;
    *p++ = G * factor;
    *p++ = B * factor;
    *p++ = pow(factor, gamma);
  }
  pTF->colorMap.finishEditing();
  return pTF;
}

